<?php
 /**
  * Model for repo_docs_clients table
  *
  * This is the model for repo_docs_clients table extended from MY_Model.
  *
  * @author Innoxess Spain, S.L. <hola@innoxess.es>
  * @copyright 2015-2018 Innoxess Spain, S.L.
  * @package  Application
  * @subpackage Models
  *
  */
if ( ! defined('BASEPATH')) exit('No direct script access allowed');
 /**
  * Class Clients_repo_docs_m - Model
  *
  * Model for repo_docs_clients table.
  */
class Clients_repo_docs_m extends MY_Model
{
	 /** @var string This is the database table for this model. */
	var $table = "repo_docs_clients";
	 /** @var int This is private var to use on this class. */
	private $doc_id;
	 /** @var int This is private var to use on this class. */
	private $client_id;

	/**
	  * Constructor
	  */
	function __construct()
	{
		parent::__construct();
		$this->load->model('repo_categories_m');
		$this->load->model('repo_docs_m');
	}

	/**
	  * Fetch all the records in the table filtered by client_id
	  *
	  * @param int $client_id Client ID
	  *
	  * @return array
	  */
	function get_by_client_id($client_id)
	{
		$this->db->where('client_id', $client_id);

		return $this->get_all();
	}

	/**
	  * Fetch a count of rows based on $repo_doc_id filter
	  *
	  * @param int $repo_doc_id Document ID
	  *
	  * @return int
	  */
	function get_by_repo_doc_id($repo_doc_id)
	{
		$query				= $this->db->where('repo_doc_id', $repo_doc_id)
							->get($this->table);

		return $query->num_rows();
	}

	/**
	  * Fetch all the external ID of clients that have an assigned specified doc
	  *
	  * @param int $repo_doc_id Document ID
	  *
	  * @return array
	  */
	function get_esb_by_repo_doc_id($repo_doc_id)
	{
		$query				= $this->db->join('clients', 'clients.id = '.$this->table.'.client_id', 'left outer')
							->select('clients.external_id as external_id')
							->where($this->table.'.repo_doc_id', $repo_doc_id)
							->get($this->table);

		return $query->result_array();
	}

	/**
	  * Fetch a single record. Returns an array
	  *
	  * @param int $client_id Client ID
	  * @param int $doc_id Document ID
	  *
	  * @return array
	  */
	function get_by_client_doc($client_id, $doc_id)
	{
		$query				= $this->db->where('client_id', $client_id)
							->where('repo_doc_id', $doc_id)
							->get($this->table);

		return $query->row_array();
	}

	/**
	  * Insert a new row into the table. $data should be an associative array of data to be inserted. Returns newly created ID.
	  *
	  * @param array $data Array
	  *
	  * @return int
	  */
	function set_client_doc($data)
	{
		$this->db->insert($this->table, $data);

		return  $this->db->insert_id();
	}

	/**
	  * Delete a row from the table by Client ID & Document ID
	  *
	  * @param int $client_id Client ID
	  * @param int $doc_id Document ID
	  *
	  * @return true
	  */
	function unset_by_client_doc($client_id, $doc_id)
	{
		$this->db->where('client_id', $client_id)
			->where('repo_doc_id', $doc_id)
			->delete($this->table);

		return true;
	}

	/**
	  * Fetch all the records in the table filtered by client_id & return an array containing repo_doc_id field of every row
	  *
	  * @param int $client_id Client ID
	  *
	  * @return array
	  */
	function get_ids_by_client_id($client_id)
	{
		$docs				= $this->get_by_client_id($client_id);
		$docs_ids			= array();
		foreach($docs as $doc)
		{
			$docs_ids[]			= $doc['repo_doc_id'];
		}

		return $docs_ids;
	}

	/**
	  * Make a multidimensional array with information about docs for a specified client ID.
	  *
	  * @param int $client_id Client ID
	  *
	  * @return array
	  */
	function get_tree($client_id)
	{
		$this->load->model('clients_m');
		$this->load->model('docs_assignment_users_m');

		$client 			= $this->clients_m->get_one($client_id);

		$docs				= $this->get_by_client_id($client_id);
		$docs_assigments	= $this->docs_assignment_users_m->get_by_client($client_id);
		foreach($docs_assigments as $doc_assigment)
		{
			$repo_docs_ids		= $this->db->select('repo_doc_id')
								->where('docs_assignment_id', $doc_assigment['docs_assignment_id'])
								->get('docs_assignment_repo_docs')
								->result_array();
			foreach($repo_docs_ids as $doc)
			{
				$docs[]			= $doc['repo_doc_id'];
			}
		}

		$docs_ids			= array();
		$cats_ids			= array();
		foreach($docs as $doc)
		{
			$docs_ids[]			= $doc['repo_doc_id'];
			$cats				= $this->repo_docs_m->get_cats_branch($doc['repo_doc_id']);
			foreach($cats as $cat_id)
			{
				if(!in_array($cat_id, $cats_ids))
				{
					$cats_ids[]			= $cat_id;
				}
			}
		}

		return $this->repo_categories_m->get_filtered_tree($cats_ids, $docs_ids);
	}

	/**
	  * Check if private vars repo_doc_id & client_id are setter to concat on data array.
	  *
	  * @param array $data Post Array
	  */
	private function save_model_data(&$data)
	{
		if(isset($this->doc_id) && !isset($data['repo_doc_id']))
		{
			$data['repo_doc_id']	= $this->doc_id;
		}
		if(isset($this->client_id) && !isset($data['client_id']))
		{
			$data['client_id']		= $this->client_id;
		}
	}

	/**
	  * Set private var category_id.
	  *
	  * @param int $doc_id Document ID
	  */
	public function set_doc_id($doc_id)
	{
		$this->doc_id = $doc_id;
	}

	/**
	  * Set private var category_id.
	  *
	  * @param int $client_id Client ID
	  */
	public function set_client_id($client_id)
	{
		$this->client_id = $client_id;
	}

	/**
	  * This callback runs before the insert of the crud.
	  * The callback takes as a 1st parameter the post array.
	  * With this opportunity you can add or unset some post variables.
	  *
	  * @param array $data Array
	  */
	function _before_insert(&$data)
	{
		$this->save_model_data($data);

		$this->load->model('repo_docs_m');

		$doc				= $this->repo_docs_m->get_one($data['repo_doc_id']);
		if($doc && $doc['push'])
		{
			$this->load->model('clients_m');

			$client				= $this->clients_m->get_one($data['client_id']);
			push_send($client['external_id'], lang("app_title") . ': ' . $data['name']);
		}
	}

	/**
	  * This callback runs before the update of the crud.
	  * The callback takes as a 1st parameter the primary key value and as 2nd the post array.
	  * With this opportunity you can add or unset some post variables.
	  *
	  * @param int $id Primary key
	  * @param array $data Array
	  */
	function _before_update($id, &$data)
	{
		$this->save_model_data($data);
	}
}