<?php 
 /**
  * Controller for Admin
  *
  * This is the controller used to Admin section.
  *
  * @author Innoxess Spain, S.L. <hola@innoxess.es>
  * @copyright 2015-2018 Innoxess Spain, S.L.
  * @package  Application
  * @subpackage Controllers
  * @api
  *
  */
if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Class Ws - Controller
 *
 * Controller for API section.
 *
 * @SWG\Resource(
 *   description="Operations about Generic API",
 *   produces="['application/json', 'application/pdf']"
 * )
 */
class Ws extends MY_Controller {

	/**
	  * Constructor
	  */
	function __construct()
	{
		parent::__construct();

		if(!defined('APP_FORK') || APP_FORK!='ws')
		{
			show_404();
		}
	}

	/**
	  * Return JSON from array
	  *
	  * @param array $data Array
	  */
	private function _out($data)
	{
		$this->load->view('output/json', array('json' => $data));
	}

	/**
	  * Make associative array from get or post params
	  *
	  * @param array $names Name of fields
	  */
	private function _get_params($names)
	{
		$values				= array();
		foreach($names as $name)
		{
			$values[$name]		= $this->input->get_post($name, TRUE);
		}

		return $values;
	}

	/**
	  * Redirect to swagger.
	  * This controller don't need index section but to prevent errors do redirection to swagger folder.
	  *
	  */
	function index()
	{
		$base				= str_replace('index.php', '', current_url());

		redirect($base.'swagger');
	}

	/**
	  * Create user
	  *
	  * This operation is used to create a user into the system.
	  *
	  * @SWG\Api(
	  *   path="/createUser",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to create a user into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="user name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="surname",
	  *	   description="Unique identifier of product from store",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="user Project ID in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function createUser()
	{
		$values				= $this->_get_params(array('id', 'name', 'surname', 'businesspartnerid'));
		$valid				= TRUE;
		foreach($values as $name=>$value)
		{
			if(($value==FALSE || empty($value)) && $name!='businesspartnerid')
			{
				$valid				= FALSE;
				break;
			}
		}
		if($valid)
		{
			$this->load->model('clients_m');

			$id					= $this->clients_m->insert([
				'external_id'		=> $values['id'],
				'name'				=> $values['name'],
				'surname'			=> $values['surname'],
				'partner_id'		=> $values['businesspartnerid'],
			]);

			if($id > 0)
			{
				$this->_out(array('status' => 'ok'));
			} else {
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'no item created'));
			}
		} else {
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id, name, surname fields are required'));
		}
	}
	/**
	  * Update user
	  *
	  * This operation is used to update a user into the system.
	  *
	  * @SWG\Api(
	  *   path="/updateUser",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a user into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="user name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="surname",
	  *	   description="Unique identifier of product from store",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="user Project ID in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function updateUser()
	{
		$values = $this->_get_params(array('id', 'name', 'surname', 'businesspartnerid'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='businesspartnerid'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			$data = array(
				'name' => $values['name'],
				'surname' => $values['surname'],
				'partner_id' => $values['businesspartnerid'],
			);
			$this->load->model('clients_m');
			$id = $this->clients_m->update_by_external_id($values['id'], $data);
			if($id>0){
				$this->_out(array('status' => 'ok'));
			}else{
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'no item updated'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id, name, surname fields are required'));
		}
	}
	/**
	  * Delete user
	  *
	  * This operation is used to update a user into the system.
	  *
	  * @SWG\Api(
	  *   path="/deleteUser",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a user into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function deleteUser()
	{
		$values = $this->_get_params(array('id'));
		if($values['id'] !== FALSE){
			$this->load->model('clients_m');
			$status = $this->clients_m->delete_external_id($values['id']);
			if($status>0){
				$this->_out(array('status' => 'ok'));
			}else{
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'no deleted items'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Create repository category
	  *
	  * This operation is used to create a Category in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_CreateCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to create a Category in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Category name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="Category Project ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Category description in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Category image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function Repository_CreateCategory()
	{
		$values = $this->_get_params(array('name', 'cat_ID', 'businesspartnerid', 'description'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='businesspartnerid'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$config['upload_path'] = BASE_UPLOAD.'image_categories/';
			$config['allowed_types'] = 'jpg|jpeg|gif|png';
			$config['encrypt_name'] = TRUE;
			$config['remove_spaces'] = TRUE;
			$config['overwrite'] = FALSE;
			
			$this->load->library('upload', $config);
			$field = "image";
			
			$dataArray = [
				'admin_id' => '0',
				'category_name' => $values['name'],
				'cat_id' => $values['cat_ID'],
				'business_partner' => $values['businesspartnerid'],
				'short_desc' => $values['description'],
				'updated_at' => date('Y-m-d H:i:s')
			];
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['image'] = '/'.$uploadedfilename;
				
			} 
			
			$this->load->model('repo_categories_m');
			$repo_cat_id = $this->repo_categories_m->setrepocategories($dataArray);
			
			$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, businesspartnerid, description fields are required'));
		}
	}
	
	/**
	  * Update repository category
	  *
	  * This operation is used to update a Category in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_UpdateCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a Category in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Category name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="Category Project ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Category description in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Category image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function Repository_UpdateCategory()
	{
		$values = $this->_get_params(array('name', 'cat_ID', 'businesspartnerid', 'description'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='businesspartnerid'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$dataArray = [
				'category_name' => $values['name'],
				'short_desc' => $values['description'],
				'updated_at' => date('Y-m-d H:i:s')
			];
			
			$config['upload_path'] = BASE_UPLOAD.'image_categories/';
			$config['allowed_types'] = 'jpg|jpeg|gif|png';
			$config['encrypt_name'] = TRUE;
			$config['remove_spaces'] = TRUE;
			$config['overwrite'] = FALSE;
			
			$this->load->library('upload', $config);
			$field = "image";
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['image'] = '/'.$uploadedfilename;
			} 
			
			$this->load->model('repo_categories_m');
			$repo_cat_update_cnt = $this->repo_categories_m->updaterepocategories($values['cat_ID'], $values['businesspartnerid'], $dataArray);
			
			if($repo_cat_update_cnt > 0)
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			else
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid cat_ID and businesspartnerid'));
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, businesspartnerid, description fields are required'));
		}
	}
	
	/**
	  * Delete repository category
	  *
	  * This operation is used to delete a Category in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_DeleteCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to delete a Category in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="Category Project ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *   )
	  * )
	  *
	  */
	function Repository_DeleteCategory()
	{
		$values = $this->_get_params(array('cat_ID', 'businesspartnerid'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='businesspartnerid'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			$this->load->model('repo_categories_m');
			$repo_cat_delete_cnt = $this->repo_categories_m->deleterepocategories($values['cat_ID'], $values['businesspartnerid']);
			
			if($repo_cat_delete_cnt > 0)
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			else
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid cat_ID and businesspartnerid'));
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, businesspartnerid, description fields are required'));
		}
	}
	
	/**
	  * Create user repository category
	  *
	  * This operation is used to create a Category in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_CreateCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to create a Category in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Category name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Category description in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Category image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function UserRepo_CreateCategory()
	{
		$values = $this->_get_params(array('name', 'cat_ID', 'userID', 'description'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='userID'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$config['upload_path'] = BASE_UPLOAD.'image_clients_categories/';
				$config['allowed_types'] = 'jpg|jpeg|gif|png';
				$config['encrypt_name'] = TRUE;
				$config['remove_spaces'] = TRUE;
				$config['overwrite'] = FALSE;
				
				$this->load->library('upload', $config);
				$field = "image";
				
				$dataArray = [
					'admin_id' => '0',
					'category_name' => $values['name'],
					'cat_id' => $values['cat_ID'],
					'client_id' => $client_id["id"],
					'short_desc' => $values['description'],
					'updated_at' => date('Y-m-d H:i:s')
				];
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['image'] = '/'.$uploadedfilename;
					
				} 
				
				$this->load->model('clients_categories_m');
				$client_cat_id = $this->clients_categories_m->setclientcategories($dataArray);
				
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));			
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, userID (MSBID), description fields are required'));
		}
	}
	
	/**
	  * Update user repository category
	  *
	  * This operation is used to update a Category in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_UpdateCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a Category in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Category name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Category description in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Category image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function UserRepo_UpdateCategory()
	{
		$values = $this->_get_params(array('name', 'cat_ID', 'userID', 'description'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='userID'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$dataArray = [
					'category_name' => $values['name'],
					'short_desc' => $values['description'],
					'updated_at' => date('Y-m-d H:i:s')
				];
				
				$config['upload_path'] = BASE_UPLOAD.'image_clients_categories/';
				$config['allowed_types'] = 'jpg|jpeg|gif|png';
				$config['encrypt_name'] = TRUE;
				$config['remove_spaces'] = TRUE;
				$config['overwrite'] = FALSE;
				
				$this->load->library('upload', $config);
				$field = "image";
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['image'] = '/'.$uploadedfilename;
				} 
				
				$this->load->model('clients_categories_m');
				$client_cat_update_cnt = $this->clients_categories_m->updateclientcategories($values['cat_ID'], $client_id["id"], $dataArray);
				
				if($client_cat_update_cnt > 0)
					$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
				else
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid cat_ID and UserID (MSBID)'));
				
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, userID (MSBID), description fields are required'));
		}
	}
	
	/**
	  * Delete user repository category
	  *
	  * This operation is used to delete a Category in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_DeleteCategory",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to delete a Category in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Category ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *   )
	  * )
	  *
	  */
	function UserRepo_DeleteCategory()
	{
		$values = $this->_get_params(array('cat_ID', 'userID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='userID'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){

			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$this->load->model('clients_categories_m');
				$repo_cat_delete_cnt = $this->clients_categories_m->deleteclientcategories($values['cat_ID'], $client_id["id"]);
				
				if($repo_cat_delete_cnt > 0)
					$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
				else
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid cat_ID and userID (MSBID)'));
				
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, userID (MSBID), description fields are required'));
		}
	}
	
	/**
	  * Create repository document
	  *
	  * This operation is used to create a Document in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_CreateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to create a Document in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Document name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="Document Project ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Document description in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Document image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="version",
	  *	   description="Document version in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="issuer",
	  *	   description="Document issuer in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="link",
	  *	   description="Document link in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="document",
	  *	   description="Document.",
	  *	   required=true,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function Repository_CreateDocument()
	{
		$values = $this->_get_params(array('name', 'businesspartnerid', 'doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && ($name!='businesspartnerid')){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$dataArray = [
				'admin_id' => '0',
				'doc_id' => $_GET['doc_ID'],
				'business_partner' => $_GET['businesspartnerid'],
				'name' => $_GET['name'],
				'last_update' => '0000-00-00',
				'order' => '0',
				'updated_at' => date('Y-m-d H:i:s')
			];
			
			if($_GET['cat_ID'] != 0){
				$this->load->model('repo_categories_m');
				$repo_cat_id = $this->repo_categories_m->getidbycategoryid($_GET['cat_ID']);
				if(count($repo_cat_id) > 0){
					$dataArray['category_id'] = $repo_cat_id['id'];
				}
			}
			
			if($_GET['description']){
				$dataArray['description'] = $_GET['description'];
			} else {
				$dataArray['description'] = '';
			}
			
			if($_GET['version']){
				$dataArray['version'] = $_GET['version'];
			} else {
				$dataArray['version'] = '';
			}
			
			if($_GET['link']){
				$dataArray['link'] = $_GET['link'];
			} else {
				$dataArray['link'] = '';
			}
			
			if($_GET['issuer']){
				$dataArray['issuer'] = $_GET['issuer'];
			} else {
				$dataArray['issuer'] = '';
			}
			
			$config['upload_path'] = BASE_UPLOAD.'image_docs/';
			$config['allowed_types'] = 'jpg|jpeg|gif|png';
			$config['encrypt_name'] = TRUE;
			$config['remove_spaces'] = TRUE;
			$config['overwrite'] = FALSE;
			
			$this->load->library('upload', $config);
			$field = "image";
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['image'] = '/'.$uploadedfilename;
			} else {
				$dataArray['image'] = "";
			}
			
			$config1['upload_path'] = BASE_UPLOAD.'docs/';
			$config1['allowed_types'] = 'jpg|jpeg|gif|png|pdf|doc|docx';
			$config1['encrypt_name'] = TRUE;
			$config1['remove_spaces'] = TRUE;
			$config1['overwrite'] = FALSE;
			
			//$this->load->library('upload', $config);
			$this->upload->initialize($config1);
			$field = "document";
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['doc'] = '/'.$uploadedfilename;
				
				$this->load->model('repo_docs_m');
				$repo_doc_id_count = $this->repo_docs_m->checkrepodocumentbydocid($_GET['doc_ID']);
				
				if($repo_doc_id_count > 0){
					$repo_doc_update_cnt = $this->repo_docs_m->updaterepodocument($_GET['doc_ID'], $dataArray);
				} else {
					$repo_doc_id = $this->repo_docs_m->setrepodocument($dataArray);
				}
				
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid information for name, cat_ID, doc_ID, businesspartnerid, and document fields.'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, doc_ID, businesspartnerid, document fields are required'));
		}
	}
	
	/**
	  * Update repository document
	  *
	  * This operation is used to update a Document in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_UpdateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a Document in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Document name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="businesspartnerid",
	  *	   description="Document Project ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Document description in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Document image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="version",
	  *	   description="Document version in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="issuer",
	  *	   description="Document issuer in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="link",
	  *	   description="Document link in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="document",
	  *	   description="Document.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function Repository_UpdateDocument()
	{
		$values = $this->_get_params(array('name', 'businesspartnerid', 'doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && ($name!='businesspartnerid')){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$dataArray = [
				'admin_id' => '0',
				'business_partner' => $_GET['businesspartnerid'],
				'name' => $_GET['name'],
				'last_update' => '0000-00-00',
				'order' => '0',
				'updated_at' => date('Y-m-d H:i:s')
			];
			
			if($_GET['cat_ID'] != 0){
				$this->load->model('repo_categories_m');
				$repo_cat_id = $this->repo_categories_m->getidbycategoryid($_GET['cat_ID']);
				if(count($repo_cat_id) > 0){
					$dataArray['category_id'] = $repo_cat_id['id'];
				}
				//$dataArray['category_id'] = $_GET['cat_ID'];
			}
			
			if($_GET['description']){
				$dataArray['description'] = $_GET['description'];
			} else {
				$dataArray['description'] = '';
			}
			
			if($_GET['version']){
				$dataArray['version'] = $_GET['version'];
			} else {
				$dataArray['version'] = '';
			}
			
			if($_GET['link']){
				$dataArray['link'] = $_GET['link'];
			} else {
				$dataArray['link'] = '';
			}
			
			if($_GET['issuer']){
				$dataArray['issuer'] = $_GET['issuer'];
			} else {
				$dataArray['issuer'] = '';
			}
			
			$config['upload_path'] = BASE_UPLOAD.'image_docs/';
			$config['allowed_types'] = 'jpg|jpeg|gif|png';
			$config['encrypt_name'] = TRUE;
			$config['remove_spaces'] = TRUE;
			$config['overwrite'] = FALSE;
			
			$this->load->library('upload', $config);
			$field = "image";
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['image'] = '/'.$uploadedfilename;
			} else {
				$dataArray['image'] = "";
			}
			
			$config1['upload_path'] = BASE_UPLOAD.'docs/';
			$config1['allowed_types'] = 'jpg|jpeg|gif|png|pdf|doc|docx';
			$config1['encrypt_name'] = TRUE;
			$config1['remove_spaces'] = TRUE;
			$config1['overwrite'] = FALSE;
			
			//$this->load->library('upload', $config);
			$this->upload->initialize($config1);
			$field = "document";
			
			if($this->upload->do_upload($field)){
				$data = $this->upload->data();
				$uploadedfilename = $data['file_name'];
				$fileoname = explode('.',$_FILES[$field]['name']);
				
				$dataArray['doc'] = '/'.$uploadedfilename;
			}
			
			$this->load->model('repo_docs_m');
			$repo_doc_update_cnt = $this->repo_docs_m->updaterepodocument($_GET['doc_ID'], $dataArray);
			
			if($repo_doc_update_cnt > 0)
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			else
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid doc_ID'));
						
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, doc_ID, businesspartnerid, document fields are required'));
		}
	}
	
	/**
	  * Delete repository document
	  *
	  * This operation is used to delete a Document in Repository into the system.
	  *
	  * @SWG\Api(
	  *   path="/Repository_deleteDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to delete a Document in Repository into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	 
	 function Repository_deleteDocument()
	{
		$values = $this->_get_params(array('doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value))){
				$valid = FALSE;
				break;
			}
		}
		if($valid){

			$this->load->model('repo_docs_m');
			$repo_doc_delete_cnt = $this->repo_docs_m->deleterepodocument($values['doc_ID']);
			
			if($repo_doc_delete_cnt > 0)
				$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
			else
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid doc_ID'));
			
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'doc_ID fields are required'));
		}
	}
	
	/**
	  * Create user repository document
	  *
	  * This operation is used to create a Document in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_CreateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to create a Document in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Document name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Document description in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Document image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="version",
	  *	   description="Document version in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="issuer",
	  *	   description="Document issuer in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="link",
	  *	   description="Document link in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="document",
	  *	   description="Document.",
	  *	   required=true,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function UserRepo_CreateDocument()
	{
		$values = $this->_get_params(array('name', 'businesspartnerid', 'userID', 'doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && ($name!='businesspartnerid')){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				$dataArray = [
					'admin_id' => '0',
					'doc_id' => $_GET['doc_ID'],
					//'business_partner' => $_GET['businesspartnerid'],
					'client_id' => $client_id["id"],
					'name' => $_GET['name'],
					'last_update' => '0000-00-00',
					'order' => '0',
					'updated_at' => date('Y-m-d H:i:s')
				];
				
				if($_GET['cat_ID'] != 0){
					$this->load->model('clients_categories_m');
					$client_cat_id = $this->clients_categories_m->getidbycategoryid($_GET['cat_ID']);
					
					if(count($client_cat_id) > 0){
						$dataArray['category_id'] = $client_cat_id['id'];
					}
				}
				
				if($_GET['description']){
					$dataArray['description'] = $_GET['description'];
				} else {
					$dataArray['description'] = '';
				}
				
				if($_GET['version']){
					$dataArray['version'] = $_GET['version'];
				} else {
					$dataArray['version'] = '';
				}
				
				if($_GET['link']){
					$dataArray['link'] = $_GET['link'];
				} else {
					$dataArray['link'] = '';
				}
				
				if($_GET['issuer']){
					$dataArray['issuer'] = $_GET['issuer'];
				} else {
					$dataArray['issuer'] = '';
				}
				
				$config['upload_path'] = BASE_UPLOAD.'image_clients_docs/';
				$config['allowed_types'] = 'jpg|jpeg|gif|png';
				$config['encrypt_name'] = TRUE;
				$config['remove_spaces'] = TRUE;
				$config['overwrite'] = FALSE;
				
				$this->load->library('upload', $config);
				$field = "image";
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['image'] = '/'.$uploadedfilename;
				} else {
					$dataArray['image'] = "";
				}
				
				$config1['upload_path'] = BASE_UPLOAD.'clients_docs/';
				$config1['allowed_types'] = 'jpg|jpeg|gif|png|pdf|doc|docx';
				$config1['encrypt_name'] = TRUE;
				$config1['remove_spaces'] = TRUE;
				$config1['overwrite'] = FALSE;
				
				$this->upload->initialize($config1);
				$field = "document";
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['doc'] = '/'.$uploadedfilename;
					
					$this->load->model('clients_docs_m');
					$client_doc_id_count = $this->clients_docs_m->checkclientdocbydocid_userid($_GET['doc_ID'], $client_id["id"]);
					
					if($client_doc_id_count > 0){
						$client_doc_update_cnt = $this->clients_docs_m->updateclientdocument($_GET['doc_ID'], $client_id["id"], $dataArray);
					} else {
						$client_doc_id = $this->clients_docs_m->setclientdocument($dataArray);
					}
					
					$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid information for name, cat_ID, doc_ID, businesspartnerid, and document fields.'));
				}
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, userID, doc_ID, businesspartnerid, document fields are required'));
		}
	}
	
	/**
	  * Update user repository document
	  *
	  * This operation is used to update a Document in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_UpdateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to update a Document in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="name",
	  *	   description="Document name in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="cat_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="description",
	  *	   description="Document description in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="image",
	  *	   description="Document image.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="version",
	  *	   description="Document version in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="issuer",
	  *	   description="Document issuer in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="link",
	  *	   description="Document link in MSP.",
	  *	   required=false,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="document",
	  *	   description="Document.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function UserRepo_UpdateDocument()
	{
		$values = $this->_get_params(array('name', 'businesspartnerid', 'userID', 'doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && ($name!='businesspartnerid')){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				$dataArray = [
					'admin_id' => '0',
					'doc_id' => $_GET['doc_ID'],
					//'business_partner' => $_GET['businesspartnerid'],
					'client_id' => $client_id["id"],
					'name' => $_GET['name'],
					'last_update' => '0000-00-00',
					'order' => '0',
					'updated_at' => date('Y-m-d H:i:s')
				];
				
				if($_GET['cat_ID'] != 0){
					$this->load->model('clients_categories_m');
					$client_cat_id = $this->clients_categories_m->getidbycategoryid($_GET['cat_ID']);
					
					if(count($client_cat_id) > 0){
						$dataArray['category_id'] = $client_cat_id['id'];
					}
				}
				
				if($_GET['description']){
					$dataArray['description'] = $_GET['description'];
				} else {
					$dataArray['description'] = '';
				}
				
				if($_GET['version']){
					$dataArray['version'] = $_GET['version'];
				} else {
					$dataArray['version'] = '';
				}
				
				if($_GET['link']){
					$dataArray['link'] = $_GET['link'];
				} else {
					$dataArray['link'] = '';
				}
				
				if($_GET['issuer']){
					$dataArray['issuer'] = $_GET['issuer'];
				} else {
					$dataArray['issuer'] = '';
				}
				
				$config['upload_path'] = BASE_UPLOAD.'image_clients_docs/';
				$config['allowed_types'] = 'jpg|jpeg|gif|png';
				$config['encrypt_name'] = TRUE;
				$config['remove_spaces'] = TRUE;
				$config['overwrite'] = FALSE;
				
				$this->load->library('upload', $config);
				$field = "image";
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['image'] = '/'.$uploadedfilename;
				} else {
					$dataArray['image'] = "";
				}
				
				$config1['upload_path'] = BASE_UPLOAD.'clients_docs/';
				$config1['allowed_types'] = 'jpg|jpeg|gif|png|pdf|doc|docx';
				$config1['encrypt_name'] = TRUE;
				$config1['remove_spaces'] = TRUE;
				$config1['overwrite'] = FALSE;
				
				$this->upload->initialize($config1);
				$field = "document";
				
				if($this->upload->do_upload($field)){
					$data = $this->upload->data();
					$uploadedfilename = $data['file_name'];
					$fileoname = explode('.',$_FILES[$field]['name']);
					
					$dataArray['doc'] = '/'.$uploadedfilename;
					
				}
				
				$this->load->model('clients_docs_m');
				$client_doc_id_count = $this->clients_docs_m->checkclientdocbydocid_userid($_GET['doc_ID'], $client_id["id"]);
				
				if($client_doc_id_count > 0){
					$client_doc_update_cnt = $this->clients_docs_m->updateclientdocument($_GET['doc_ID'], $client_id["id"], $dataArray);
					$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID) and doc_ID.'));
				}
				
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, cat_ID, userID, doc_ID, businesspartnerid, document fields are required'));
		}
	}
	/**
	  * Delete user repository document
	  *
	  * This operation is used to delete a Document in UserRepo into the system.
	  *
	  * @SWG\Api(
	  *   path="/UserRepo_DeleteDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to delete a Document in UserRepo into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function UserRepo_DeleteDocument()
	{
		$values = $this->_get_params(array('userID', 'doc_ID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value))){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			//var_dump("Under Development");exit();
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$this->load->model('clients_docs_m');
				$client_doc_delete_cnt = $this->clients_docs_m->deleteclientdocument($values['doc_ID'], $client_id["id"]);
				
				if($client_doc_delete_cnt > 0)
					$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
				else
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid doc_ID, and UserID (MSBID)'));
				
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'userID, and doc_ID fields are required'));
		}
	}

	/**
	  * Get documents info
	  *
	  * This operation is used to get all the data information of the documents of a user. 
	  * Contains document details and categories.
	  *
	  * @SWG\Api(
	  *   path="/getDocsInfo",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to get all the data information of the documents of a user. Contains document details and categories.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function getDocsInfo()
	{
		$values = $this->_get_params(array('id'));
		if($values['id'] !== FALSE){
			$this->load->model('clients_m');
			$client = $this->clients_m->get_by_external_id($values['id']);
			if(isset($client['id'])){
				$this->load->model('clients_categories_m');
				$this->load->model('clients_repo_docs_m');
				$usertree = $this->clients_categories_m->get_tree($client['id']);
				$repotree = $this->clients_repo_docs_m->get_tree($client['id']);
				
				/*foreach($usertree as $key=>$value){
					unset($usertree[$key]);
					if($key==="docs"){
						$usertree[$key] = $value;
					}else{
						$usertree[$key."_"] = $value;
					}
				}
				
				foreach($repotree as $key=>$value){
					unset($repotree[$key]);
					if($key==="docs"){
						$repotree[$key] = $value;
					}else{
						$repotree[$key."_"] = $value;
					}
				} */
				
				$this->_out(array('status' => 'ok', 'usertree' => $usertree, 'repotree' => $repotree));
			}else{
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'client not exist'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Get Document info.
	  *
	  * This operation is used to get one document from one user.
	  *
	  * @SWG\Api(
	  *   path="/getDocInfo",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to get one document from one user.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="doc_id",
	  *	   description="Document identifier.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function getDocInfo()
	{
		$values = $this->_get_params(array('id', 'doc_id'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if($value==FALSE || empty($value)){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			switch (substr($values['doc_id'], 0, 3)) {
				case 'dr_':
					$values['doc_id'] = str_replace('dr_', '', $values['doc_id']);
					$dir = 'docs';
					$this->load->model('repo_docs_m');
					$doc = $this->repo_docs_m->get_one($values['doc_id']);
					$file = $doc['doc'];
					break;
				case 'dc_':
					$values['doc_id'] = str_replace('dc_', '', $values['doc_id']);
					$dir = 'clients_docs';
					$this->load->model('clients_docs_m');
					$doc = $this->clients_docs_m->get_one($values['doc_id']);
					$file = $doc['doc'];
					break;
				default:
					$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'doc_id not correct'));
					break;
			}
			if(isset($dir)){
				$this->load->helper('file');
				$content = read_file(BASE_UPLOAD.$dir.'/'.$file);
				if($content != FALSE){
					$this->output->set_content_type(get_mime_by_extension($file));
					$this->output->set_output($content);
				}else{
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
				}
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id, doc_id fields are required'));
		}
	}
	
	/**
	  * Get image.
	  *
	  * This operation is used to get associated image of a document or category from one user.
	  *
	  * @SWG\Api(
	  *   path="/getImage",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to get associated image of a document or category from one user.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="User identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="item_id",
	  *	   description="Document/item identifier.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function getImage()
	{
		$values = $this->_get_params(array('id', 'item_id'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if($value==FALSE || empty($value)){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			switch (substr($values['item_id'], 0, 3)) {
				case 'cr_':
					$values['item_id'] = str_replace('cr_', '', $values['item_id']);
					$dir = 'image_categories';
					$this->load->model('repo_categories_m');
					$cat = $this->repo_categories_m->get_one($values['item_id']);
					$file = $cat['image'];
					$defaultDir = 'image_categories';
					$defaultFile = 'defaultchildcat.png';
					break;
				case 'dr_':
					$values['item_id'] = str_replace('dr_', '', $values['item_id']);
					$dir = 'image_docs';
					$this->load->model('repo_docs_m');
					$doc = $this->repo_docs_m->get_one($values['item_id']);
					$file = $doc['image'];
					$defaultDir = 'image_docs';
					$defaultFile = 'defaultdocimg.png';
					break;
				case 'cc_':
					$values['item_id'] = str_replace('cc_', '', $values['item_id']);
					$dir = 'image_clients_categories';
					$this->load->model('clients_categories_m');
					$cat = $this->clients_categories_m->get_one($values['item_id']);
					$file = $cat['image'];
					$defaultDir = 'image_categories';
					$defaultFile = 'defaultchildcat.png';
					break;
				case 'dc_':
					$values['item_id'] = str_replace('dc_', '', $values['item_id']);
					$dir = 'image_clients_docs';
					$this->load->model('clients_docs_m');
					$doc = $this->clients_docs_m->get_one($values['item_id']);
					$file = $doc['image'];
					$defaultDir = 'image_docs';
					$defaultFile = 'defaultdocimg.png';
					break;
				default:
					$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'item_id not correct'));
					break;
			}
			if(isset($dir)){
				$this->load->helper('file');
				$content = read_file(BASE_UPLOAD.$dir.'/'.$file);
				if($content == FALSE){
					$content = read_file(BASE_UPLOAD.$defaultDir.'/'.$defaultFile);
				}
				if($content != FALSE){
					$this->output->set_content_type(get_mime_by_extension($file));
					$this->output->set_output($content);
				}else{
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
				}
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id, item_id fields are required'));
		}
	}
	
	/**
	  * Get private image
	  *
	  * This operation is used to get private image of a project from one user.
	  *
	  * @SWG\Api(
	  *   path="/getPrivateImage",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to get private image of a project from one user.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="User identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function getPrivateImage()
	{
		$values				= $this->_get_params(array('id'));
		$valid				= TRUE;
		foreach($values as $name=>$value)
		{
			if($value==FALSE || empty($value))
			{
				$valid = FALSE;
				break;
			}
		}
		if($valid)
		{
			$this->load->model('clients_m');

			$client				= $this->clients_m->get_by_external_id($values['id']);
			if(isset($client['partner_id']))
			{
				$this->load->model('business_partners_m');

				$partner			= $this->business_partners_m->get_by_external_id($client['partner_id']);
				if(!is_null($partner['general_image']))
				{
					$this->load->helper('file');

					$content			= read_file(BASE_UPLOAD.'business_partners/'.$partner['general_image']);
					if($content != FALSE)
					{
						$this->output->set_content_type(get_mime_by_extension($partner['general_image']));
						$this->output->set_output($content);
					} else {
						$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
					}
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
				}
			} else {
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'id not correct'));
			}
		} else {
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Get general image.
	  *
	  * This operation is used to get general image of a project from one user.
	  *
	  * @SWG\Api(
	  *   path="/getGeneralImage",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to get general image of a project from one user.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="User identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function getGeneralImage()
	{
		$values				= $this->_get_params(array('id'));
		$valid				= TRUE;
		foreach($values as $name=>$value)
		{
			if($value==FALSE || empty($value))
			{
				$valid = FALSE;
				break;
			}
		}
		if($valid)
		{
			$this->load->model('clients_m');

			$client				= $this->clients_m->get_by_external_id($values['id']);
			if(isset($client['partner_id']))
			{
				$this->load->model('business_partners_m');

				$partner			= $this->business_partners_m->get_by_external_id($client['partner_id']);
				if(!is_null($partner['general_document_image']))
				{
					$this->load->helper('file');

					$content			= read_file(BASE_UPLOAD.'business_partners/'.$partner['general_document_image']);
					if($content != FALSE)
					{
						$this->output->set_content_type(get_mime_by_extension($partner['general_document_image']));
						$this->output->set_output($content);
					} else {
						$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
					}
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'file not found'));
				}
			} else {
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'id not correct'));
			}
		} else {
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Upload invoices
	  *
	  * This operation is used to upload an invoices into the system.
	  *
	  * @SWG\Api(
	  *   path="/uploadInvoices",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to upload an invoices into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	   @SWG\Parameter(
	  *	   name="invoice",
	  *	   description="select invoice.",
	  *	   required=false,
	  *	   type="file",
	  *	   paramType="form"
	  *	 )
	  *   )
	  * )
	  *
	  */
	function uploadInvoices()
	{
		$values = $this->_get_params(array('id'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if($value==FALSE || empty($value)){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client = $this->clients_m->get_by_external_id($values['id']);
			
			if(isset($client['id'])){
					$this->load->model('clients_categories_m');
					$categoryid = $this->clients_categories_m->getinvoice($client['id']);
					
					if($categoryid > 0){
						
						$config['upload_path'] = BASE_UPLOAD.'clients_docs/';
						//$config['upload_path'] = TEMP_UPLOAD;
						$config['allowed_types'] = 'jpg|jpeg|gif|png|pdf|doc|docx';
						$config['encrypt_name'] = TRUE;
						$config['remove_spaces'] = TRUE;
						$config['overwrite'] = FALSE;
						
						$this->load->library('upload', $config);
						$field = "invoice";
						if(!empty($field)){
							
							if($this->upload->do_upload($field)){
								$data = $this->upload->data();
								$uploadedfilename = $data['file_name'];
								$fileoname = explode('.',$_FILES[$field]['name']);
								$this->load->model('clients_docs_m');
								$docid = $this->clients_docs_m->setinvoice($client['id'], $categoryid, $uploadedfilename, $fileoname[0]);
								
								$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => "success"));
								
							} else{
								$msg = $this->upload->display_errors('<p>', '</p>');
								$this->_out(array('status' => 'error', 'code' => -3, 'msg' => $msg));
							}
						} else{
							$this->_out(array('status' => 'error', 'code' => -3, 'msg' => "Please select Invoice"));
						}				
					} else {
						$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'Please create Invoice'));
					}
			} else {
					$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'client not exist'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id, invoice fields are required'));
		}
	}
	
	/**
	  * Activate invoice
	  *
	  * This operation is used to activate invoice into the system.
	  *
	  * @SWG\Api(
	  *   path="/activateInvoice",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to activate invoice into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function activateInvoice()
	{
		$values = $this->_get_params(array('id'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if($value==FALSE || empty($value)){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			$this->load->model('clients_m');
			$client = $this->clients_m->get_by_external_id($values['id']);
			
			if(isset($client['id'])){
				
				$this->load->model('clients_categories_m');
				$categoryid = $this->clients_categories_m->createinvoice($client['id']);
				
				if($categoryid > 0){
					$this->_out(array('status' => 'ok'));
				}else{
					$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'Something went wrong'));
				}
			}else{
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'client not exist'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Deactivate invoice
	  *
	  * This operation is used to deactivate invoice into the system.
	  *
	  * @SWG\Api(
	  *   path="/deactivateInvoice",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to deactivate invoice into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="id",
	  *	   description="user identifier in MSP.",
	  *	   required=true,
	  *	   type="integer",
	  *	   format="integer",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 )
	  *   )
	  * )
	  *
	  */
	function deactivateInvoice()
	{
		$values = $this->_get_params(array('id'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if($value==FALSE || empty($value)){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			$this->load->model('clients_m');
			$client = $this->clients_m->get_by_external_id($values['id']);
			
			if(isset($client['id'])){
				
				$this->load->model('clients_categories_m');
				$categoryid = $this->clients_categories_m->deactivateinvoice($client['id']);
				
				if($categoryid > 0){
					$this->_out(array('status' => 'ok'));
				}else{
					$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'Please create Invoice'));
				}
			}else{
				$this->_out(array('status' => 'error', 'code' => -2, 'msg' => 'client not exist'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'id field is required'));
		}
	}
	
	/**
	  * Associate Document
	  *
	  * This operation is used to associate Document to user into the system.
	  *
	  * @SWG\Api(
	  *   path="/associateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to associate Document to user into the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *   )
	  * )
	  *
	  */
	function associateDocument()
	{
		$values = $this->_get_params(array('doc_ID', 'userID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='userID'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$this->load->model('repo_docs_m');
				$doc_info = $this->repo_docs_m->get_by_doc_id($values['doc_ID']);
				
				if(count($doc_info) > 0){
					
					$this->load->model('clients_repo_docs_m');
					$client_doc_info = $this->clients_repo_docs_m->get_by_client_doc($client_id['id'], $doc_info['id']);
					
					if(count($client_doc_info) > 0){
						$this->_out(array('status' => 'existing', 'code' => -2, 'msg' => 'existing'));
					} else {
						$dataArray = [
							'repo_doc_id' => $doc_info['id'],
							'client_id' => $client_id['id'],
						];
						
						$client_doc_id = $this->clients_repo_docs_m->set_client_doc($dataArray);
						$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => 'ok'));
					}
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid doc_ID.'));
				}
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, doc_ID, userID (MSBID) fields are required'));
		}
	}
	
	/**
	  * Disassociate document
	  *
	  * This operation is used to disassociate Document to user from the system.
	  *
	  * @SWG\Api(
	  *   path="/disassociateDocument",
	  *   @SWG\Operation(
	  *	 method="POST",
	  *	 summary="This operation is used to disassociate Document to user from the system.",
	  *	 notes="",
	  *	 type="void",
	  *	 @SWG\Parameter(
	  *	   name="doc_ID",
	  *	   description="Document ID in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *	 @SWG\Parameter(
	  *	   name="userID",
	  *	   description="userID (MSPID) in MSP.",
	  *	   required=true,
	  *	   type="string",
	  *	   format="string",
	  *	   paramType="query",
	  *	   allowMultiple=false
	  *	 ),
	  *   )
	  * )
	  *
	  */
	function disassociateDocument()
	{
		$values = $this->_get_params(array('doc_ID', 'userID'));
		$valid = TRUE;
		foreach($values as $name=>$value){
			if(($value==FALSE || empty($value)) && $name!='userID'){
				$valid = FALSE;
				break;
			}
		}
		if($valid){
			
			$this->load->model('clients_m');
			$client_id = $this->clients_m->get_by_external_id($values['userID']);
			
			if(count($client_id) > 0){
				
				$dataArray = [
					'repo_doc_id' => $values['doc_ID'],
					'client_id' => $client_id['id'],
				];
				
				$this->load->model('repo_docs_m');
				$doc_info = $this->repo_docs_m->get_by_doc_id($values['doc_ID']);
				
				if(count($doc_info) > 0){
					
					$this->load->model('clients_repo_docs_m');
					$client_doc_info = $this->clients_repo_docs_m->get_by_client_doc($client_id['id'], $doc_info['id']);
					if(count($client_doc_info) > 0){
						$client_doc_delete = $this->clients_repo_docs_m->unset_by_client_doc($client_id['id'], $doc_info['id']);
						$this->_out(array('status' => 'ok', 'code' => 1, 'msg' => 'ok'));
					} else {
						$this->_out(array('status' => 'not exist', 'code' => -2, 'msg' => 'not exist'));
					}
				} else {
					$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid doc_ID.'));
				}
			} else {
				$this->_out(array('status' => 'error', 'code' => -3, 'msg' => 'Please provide valid UserID (MSBID).'));
			}
		}else{
			$this->_out(array('status' => 'error', 'code' => -1, 'msg' => 'name, doc_ID, userID (MSBID) fields are required'));
		}
	}
}